<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Traits\EmailTrait;
use App\Models\TierPackage;
use App\Models\LeavePackage;
use App\Models\LeaveRequest;
use App\Models\RosterAssign;
use Illuminate\Http\Request;
use App\Models\EmpCompanyDetails;
use App\Models\EmployeeAttendance;
use App\Models\EmpPersonalDetails;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class LeaveRequestController extends Controller
{   
    use EmailTrait;
    public function __construct()
    {
        $this->middleware('Permissions:Leave Request Maintain')->only(['create', 'edit']);
    }
    public function index(Request $request)
    {
        $query = LeaveRequest::query();

        if($request->filled('filter'))
        {
          $query =  $this->filter(json_decode($request->filter, true),$query);
        }
        $query->with('LeavePackage')->with('EmpPersonalDetails');

        $count = $query->count();
        $start = $request->input('from', 0);
        $query->offset($start)->limit(10);
        $query->orderBy('id', 'desc');
        $query_result = $query->get();

        $data = [
            'leaverequest' => $query_result,
        ];
        
        $array_filter =  json_decode($request->filter, true);
        $filters = [
          'first_name' => $array_filter['first_name'] ?? '', 
          'middle_name' => $array_filter['middle_name'] ?? '',
          'last_name' => $array_filter['first_name'] ?? '',
          'employee_email' => $array_filter['last_name'] ?? '',
          'compeleted' => $array_filter['first_name'] ?? '',
          'status' => $array_filter['compeleted'] ?? '',
        ];

        return response()->json([
            'message' => 'Get Leave Package List Successfully',
            'data' => $data,
            'count' => $count,
            'filters' => $filters
        ],200);
    }

    public function filter($filters,$query)
    {
        foreach ($filters as $filterName => $filterValue) {
              if ($filterValue != null ||  $filterValue !="" ) {
                  switch ($filterName) {
                    case 'first_name':
                        $filterValue  = explode(" ",$filterValue);
                        if(count($filterValue) == 1)
                        {
                            $query->whereHas('empDetails', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                            });
                        }
    
                        if(count($filterValue) == 2)
                        {
                            $query->whereHas('empDetails', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                                $subquery->where('last_name', 'like', '%' . $filterValue[1] . '%');
                            });
                        }
    
                        if(count($filterValue) == 3)
                        {
                            $query->whereHas('empDetails', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%');
                                $subquery->where('middle_name', 'like', '%' . $filterValue[1] . '%');
                                $subquery->where('last_name', 'like', '%' . $filterValue[2] . '%');
                            });
                        }
    
                      break;
                  case 'leave_package_id':
                      $query->where('leave_package_id',$filterValue);
                  break;
                  case 'to':
                    $query->where('to',date('Y-m-d',strtotime($filterValue)));
                break;
                case 'from':
                    $query->where('from',date('Y-m-d',strtotime($filterValue)));
                break;
                  case 'status':
                      $query->where('status',$filterValue);
                  break;                 
                  }
              }
        }

        return $query;
    } 


    public function create()
    {
         $leave_packages =  LeavePackage::orderBy('id', 'DESC')->get();
      
        $emp_details =  EmpCompanyDetails::with('EmpPersonalDetails')->where('del',0)->where('status',1)->where('compeleted', 1)->orderBy('id', 'DESC')->get();
        return view('LeaveRequest.create',compact('leave_packages','emp_details'));
    }

   
    public function store(Request $request)
    {
        $employee = EmpCompanyDetails::find($request->employee);  

        if (!$employee) {
            return response()->json([
                'message' => 'Employee not found'
            ], 404);
        }

        if ($employee->status == '0') {
            return response()->json([
                'message' => 'The employee account is currently inactive'
            ], 400);
        }

        if ($employee->del == '1') {
            return response()->json([
                'message' => 'The employee account has been deleted'
            ], 400);
        }
        $validator = Validator::make($request->all(), [
            'employee' => 'required',
            'leave_package' => 'required',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'reason' => 'required',
            'full_day' => 'nullable',
            'start_time' => 'nullable',
            'end_time' => 'nullable'
        ]);
 
        if ($validator->fails()) {

            $error = $validator->errors()->first();

            return response()->json([
                'message' =>  $error
            ],422);

        } else {
          
        $validatedData = $validator->validated();
        $fromDate = Carbon::parse($validatedData['start_date']);
        $toDate = Carbon::parse($validatedData['end_date']);
        $numberOfDays = $fromDate->diffInDays($toDate) + 1; // Include start and end dates
        if(
            (LeaveRequest::where('from','>=',$validatedData['start_date'] )->where('status','!=','2')->where('employee_id',$validatedData['employee'])->count()
             && LeaveRequest::where('to','<=',$validatedData['end_date'] )->where('status','!=','2')->where('employee_id',$validatedData['employee'])->count())
            ||LeaveRequest::where('from',$validatedData['start_date'] )->where('status','!=','2')->where('employee_id',$validatedData['employee'])->count()
          )
        {
            return response()->json([
                'message' => 'You already have leave scheduled for this date',
                'data'   => []
            ],400);
        }
      
        $insertedRecord = LeaveRequest::create([
            'employee_id' =>$validatedData['employee'],
            'leave_package_id' => $validatedData['leave_package'],
            'from' => $validatedData['start_date'],
            'to' => $validatedData['end_date'],
            'reason' => $validatedData['reason'],
            'full_day' => $validatedData['full_day'],
            'start_time' => $validatedData['start_time'],
            'number_of_leaves' => $numberOfDays,
            'end_time' => $validatedData['end_time'],
            'applied_by' => Auth::user()->id
        ]);
        
        $id = $insertedRecord->id;
        $data = LeaveRequest::find($id);
        session()->flash('success', 'Leave Request Saved Successfully');
            return response()->json([
                'message' => 'Leave Request Saved Successfully',
                'data'   => $data
            ],200);
            
        } 
    }
    ////////////////leave Request Api///////////////
    public function applie_leave_request(Request $request)
{
    $validator = Validator::make($request->all(), [
        'employee_id'       => 'required',
        'leave_package_id'  => 'required',
        'from'              => 'required|date',
        'mail_to'           => 'required',
        'mail_cc'           => 'required|array',
        'to'                => 'required|date|after_or_equal:from',
        'reason'            => 'required'
    ], [
        // Here you specify your custom messages
        'from.required' => 'Start date is required.',  // Custom message for `from` field
        'to.required'   => 'End date is required.',    // Custom message for `to` field
    ]);

    if ($validator->fails()) {
        return response()->json([
            'message' => $validator->errors()->first()
        ], 422);
    }
    $validatedData = $validator->validated();

    // Calculate number of leave days
    $fromDate = Carbon::parse($validatedData['from']);
    $toDate = Carbon::parse($validatedData['to']);
    $numberOfDays = $fromDate->diffInDays($toDate) + 1; // Include start and end dates
    // Check if leave overlaps with existing leaves
    $existingLeaves = LeaveRequest::where('employee_id', $validatedData['employee_id'])
        ->where('status', '!=', 2)
        ->where(function ($query) use ($validatedData) {
            $query->whereBetween('from', [$validatedData['from'], $validatedData['to']])
                ->orWhereBetween('to', [$validatedData['from'], $validatedData['to']])
                ->orWhere(function ($q) use ($validatedData) {
                    $q->where('from', '<=', $validatedData['from'])
                        ->where('to', '>=', $validatedData['to']);
                });
        })
        ->exists();

    if ($existingLeaves) {
        return response()->json([
            'message' => 'You already have a leave scheduled during this period'
        ], 400);
    }

    // Fetching details
    $leavePackage = LeavePackage::find($validatedData['leave_package_id']);
    $empDetail = EmpPersonalDetails::where('emp_id',$validatedData['employee_id'])->first();
    $mailTo = EmpCompanyDetails::find($validatedData['mail_to'])->employee_email;
    $employee = EmpCompanyDetails::with('empTeamsMembers.empTeamsList')
        ->where('id', $validatedData['employee_id'])
        ->first();
    $teamTitle = null;
    if ($employee) {
        foreach ($employee->empTeamsMembers as $teamMember) {
            if (isset($teamMember->empTeamsList)) {
                $teamTitle = $teamMember->empTeamsList->title;
                break;
            }
        }
    }

    // Handling CCs
    $ccEmails = EmpCompanyDetails::whereIn('id', $validatedData['mail_cc'])->pluck('employee_email')->toArray();

    // Email data
    $emailData = [
        'name' => $empDetail->first_name . ' ' . $empDetail->middle_name . ' ' . $empDetail->last_name,
        'leave_pkg' => $leavePackage->title,
        'team' => $teamTitle,
        'start_date' => $validatedData['from'],
        'end_date' => $validatedData['to'],
        'reason' => $validatedData['reason']
    ];

    // Send email with CC support
    $emailSent = $this->sendEmail_cc($mailTo, $leavePackage->title . ' | ' . env("APP_NAME"), $emailData, 'Emails.Leave-Request', $ccEmails);

    // Store leave request
    $validatedData['number_of_leaves'] = $numberOfDays;
    $insertedRecord = LeaveRequest::create($validatedData);
    $data = LeaveRequest::find($insertedRecord->id);

    return response()->json([
        'message' => 'Leave Saved Successfully',
        'email_status' => $emailSent ? 'Email sent successfully.' : 'Email sending failed.',
        'data' => $data
    ], 200);
}


    public function get_leave_request(Request $request)
    {
        
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required'
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
            return response()->json([
                'message' => $error
            ], 422);
        } else {
            $validatedData = $validator->validated();
           
            $data = LeaveRequest::where('employee_id', $validatedData['employee_id'])
            ->with(['empCompanyDetails.empPersonalDetails'])
            ->with('LeavePackage')
            ->get();

            if ($data->isNotEmpty()) {
                return response()->json([
                    'message' => 'Leave Retrieved Successfully',
                    'data'   => $data
                ], 200);
            } else {
                return response()->json([
                    'message' => 'Leave not found for the given employee',
                ], 200);
            }
        }
    }

    public function edit_leave_request(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'leave_id' => 'required'
        ]);
    
        if ($validator->fails()) {
            $error = $validator->errors()->first();
            return response()->json([
                'message' => $error
            ], 422);
        } else {
            $validatedData = $validator->validated();
            $data = LeaveRequest::where('id', $validatedData['leave_id'])->first();
    
            if ($data !== null) {
                return response()->json([
                    'message' => 'Leave Retrieved Successfully',
                    'data'   => $data
                ], 200);
            } else {
                return response()->json([
                    'message' => 'Leave not found',
                ], 404);
            }
        }
    }

    public function update_leave_request(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required',
            'leave_package_id' => 'required',
            'from' => 'required|date',
            'to' => 'required|date|after_or_equal:from',
            'reason' => 'required'
        ]);

        if ($validator->fails()) {
            $error = $validator->errors()->first();
            return response()->json([
                'message' => $error
            ], 422);
        } else {
            $validatedData = $validator->validated();

            $leaveRequest = LeaveRequest::find($id);

            if ($leaveRequest) {
                $leaveRequest->update($validatedData);

                $data = LeaveRequest::find($id);
                return response()->json([
                    'message' => 'Leave Updated Successfully',
                    'data'   => $data
                ], 200);
            } else {
                return response()->json([
                    'message' => 'Leave not found',
                ], 404);
            }
        }
    }

    public function employee_leave_pkg(Request $request)
    {
        
        $emp_details = EmpCompanyDetails::where('id', $request->employee_id)->first();
        $leave_packages    = LeavePackage::whereRaw("FIND_IN_SET(?, employee_roles)", [$emp_details->accessRole->id])->get();
        

        if (!$leave_packages) {
            return response()->json(['error' => 'Leave package not found'], 400);
        }
    
  
        return response()->json([
            'all_packages' => $leave_packages
        ], 200);
    }

////////////////leave Request Api///////////////

      
    

    public function edit($id)
    {
        $leave_packages =  LeavePackage::orderBy('id', 'DESC')->get();
        $emp_details =  EmpCompanyDetails::with('EmpPersonalDetails')->where('del',0)->where('compeleted', 1)->orderBy('id', 'DESC')->get();
        $leaverquest = LeaveRequest::where('id',$id)->first();
        return view('LeaveRequest.edit', compact('leave_packages', 'emp_details','leaverquest'));
    }


    public function edit_leave_for_roster(Request $request)
    {

        $leave_rquest    =   LeaveRequest::where('id',$request->id)->first();

        $data['leave_rquest'] = $leave_rquest;

        return response()->json([
            'message' => 'Get Leave Successfully',
            'data'   => $data
        ],200);
    }

    public function update(Request $request, $id)
    {
       
        
        $validator = Validator::make($request->all(), [
            'employee' => 'required',
            'leave_package' => 'required',
            'start_date' => 'required|date',
            'end_date' => 'required|date|after_or_equal:start_date',
            'reason' => 'required',
            'full_day' => 'nullable',
            'start_time' => 'nullable',
            'end_time' => 'nullable'
        ]);
 
        if ($validator->fails()) {

            $error = $validator->errors()->first();

            return response()->json([
                'message' => $error

            ],422);

        } else {
          
            $user = Auth::user();

            $validatedData = $validator->validated();
            $validatedData['applied_by'] = $user->id;
            $fromDate = Carbon::parse($validatedData['start_date']);
            $toDate = Carbon::parse($validatedData['end_date']);
            $numberOfDays = $fromDate->diffInDays($toDate) + 1; // Include start and end dates
            $leaveRequest = LeaveRequest::findOrFail($id);
            $leaveRequest->update([
                'employee_id' =>$validatedData['employee'],
                'leave_package_id' => $validatedData['leave_package'],
                'from' => $validatedData['start_date'],
                'to' => $validatedData['end_date'],
                'reason' => $validatedData['reason'],
                'full_day' => $validatedData['full_day'] ?? 0,
                'number_of_leaves' => $numberOfDays,
                'start_time' => $validatedData['start_time'],
                'end_time' => $validatedData['end_time']
            ]);

           $data = LeaveRequest::where('id',$id)->get();
           session()->flash('success', 'Leave Request Update Successfully');
            return response()->json([
                'message' => 'Leave Request Update Successfully',
                'data'   => $data
            ],200);
            
        } 
    }

    public function destroy($id)
    {
        LeaveRequest::where('id',$id)->delete();
        session()->flash('success', 'Leave Request Deleted Successfully');
        return response()->json([
            'message' => 'Leave Request Deleted Successfully'
        ],200);
    }

    public function updateStatus(Request $request, $id)
    {
        $this->validate($request, [
            'status' => 'required|integer|in:1,2,4',
            'reason' => 'nullable|string'
        ]);
        $leaveRequest = LeaveRequest::findOrFail($id);
        $leaveRequest->status = $request->input('status');
        $leaveRequest->approved_by = Auth::user()->id;
        
        // Handle different status types
        if ($leaveRequest->status == 2) {
            // Rejected
            $leaveRequest->custom_approval_reason = $request->input('reason');
        } elseif ($leaveRequest->status == 4) {
            // Approved as Unpaid - set paid_days to 0 and paid_dates to empty
            $leaveRequest->paid_days = 0;
            $leaveRequest->paid_dates = json_encode([]);
            $leaveRequest->custom_approval_reason = $request->input('reason');
        }
        
        $leaveRequest->save();
        $leave_pkg = LeavePackage::where('id',$leaveRequest->leave_package_id)->first();
        $emp_mail = EmpCompanyDetails::where('id',$leaveRequest->employee_id)->first();
        $emp_detail = EmpPersonalDetails::where('emp_id',$leaveRequest->employee_id)->first();

        $data = [
            'status' => $leaveRequest->status,
            'name' => $emp_detail->first_name . ' ' . $emp_detail->middle_name . ' ' . $emp_detail->last_name,
            'subject' => $leave_pkg->title.' | '.env("APP_NAME"),
            'reason' => $leaveRequest->custom_approval_reason ?? ''
        ];

        // Send email using the EmailTrait function
       


        if( $leaveRequest->status == 1 ||  $leaveRequest->status == 2 || $leaveRequest->status == 4){
             $emailSent = $this->sendEmail(
                $emp_mail->employee_email, 
                $leave_pkg->title.' | '.env("APP_NAME"), 
                $data, 
                'Emails.leave-email'
            );
            
            return response()->json(['message' => 'Status updated successfully and email has been send']);
            
        }else{
            return response()->json(['message' => 'Status updated successfully']);
        }
       

       
    }

    public function leavereason($id)
    {
        $data = LeaveRequest::findOrFail($id);
        
        return response()->json([
            'data'   => $data
        ],200);
    }

    public function remainingleave(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'employee_id' => 'required|integer',
            'leave_package_id' => 'required|integer',
        ]);

        if ($validator->fails()) {
            $error = $validator->errors()->first();
            return response()->json([
                'message' => $error

            ], 422);
        } else {
            $leave_request = LeaveRequest::where('leave_package_id', $request->leave_package_id)->where('employee_id',$request->employee_id)->first();

            if (!$leave_request) {
                return response()->json([
                    'message' => 'Leave request not found',
                ], 404);
            }

            $from = new \DateTime($leave_request->from);
            $to = new \DateTime($leave_request->to);

            $no_of_leaves = $from->diff($to)->days + 1;


            $_leave_pkg = LeavePackage::where('id', $request->leave_package_id)->first();

            if (!$_leave_pkg) {
                return response()->json([
                    'message' => 'Leave package not found',
                ], 404);
            }

            $total_pkg_no = $_leave_pkg->leave_number;

            $remaining_leave = $total_pkg_no - $no_of_leaves;
            return response()->json([
                'message' => 'Remaining Leave',
                'data'   => $remaining_leave
            ], 200);
        }
    }

    public function view($id)
    {
        $leave_request = LeaveRequest::where('id', $id)->first();
        $leave_pgk = LeavePackage::where('id', $leave_request->leave_package_id)->first();
        $emp_details = EmpCompanyDetails::with('EmpPersonalDetails')->where('id', $leave_request->employee_id)->first();        
        $all_pkgs = LeavePackage::whereRaw("FIND_IN_SET(?, employee_roles)", [$emp_details->accessRole->id])->get();

        $emp_role = LeaveRequest::where('leave_requests.id', $id)
        ->join('emp_company_details', 'leave_requests.employee_id', '=', 'emp_company_details.id')
        ->join('roles', 'emp_company_details.access_role', '=', 'roles.code')
        ->select('roles.*')
        ->first();

        $currentDate = Carbon::today();
        $today_on_leave_employes = DB::table('leave_requests')
        ->where('leave_requests.status',1)
        ->whereDate('leave_requests.created_at', $currentDate)
        ->join('emp_company_details', 'leave_requests.employee_id', '=', 'emp_company_details.id')
        ->join('emp_personal_details','leave_requests.employee_id', '=', 'emp_personal_details.emp_id')
        ->join('roles', 'emp_company_details.access_role', '=', 'roles.code')
        ->select('roles.title', 'emp_personal_details.*')
        ->get();
        return view('LeaveRequest.view', compact('leave_request', 'emp_details', 'leave_pgk', 'all_pkgs','emp_role','today_on_leave_employes'));
    }
    
    public function viewLeaves(Request $request)
    {
        $authUserId = Auth::id();
        $emp_details = EmpCompanyDetails::with('EmpPersonalDetails', 'accessRole')->findOrFail($authUserId);
    
        // Fetch leave requests of the employee
        $leave_requests = LeaveRequest::where('employee_id', $authUserId)->get();
    
        // Fetch leave packages based on employee role
        $all_pkgs = LeavePackage::whereRaw("FIND_IN_SET(?, employee_roles)", [$emp_details->accessRole->id])->get();
    
        // Process leave packages
        $all_pkgs = $all_pkgs->map(function ($pkg) use ($emp_details) {
            $takenLeaves = DB::table('leave_requests')
                ->where('leave_package_id', $pkg->id)
                ->where('employee_id', $emp_details->id)
                ->where('status', 1) // Only count approved (1) leaves for balance, exclude approved as unpaid (4)
                ->sum('number_of_leaves'); // Sum up taken leaves
    
            $pkg->approved_leave_count = $takenLeaves;
            $pkg->remaining_leaves = $pkg->leave_number - $takenLeaves; // Calculate remaining leaves
            return $pkg;
        });
    
        // Get today's date
        $currentDate = Carbon::today();
    
        // Get employees on leave today
        $today_on_leave_employees = DB::table('leave_requests')
            ->where('leave_requests.status', 1)
            ->whereDate('leave_requests.created_at', $currentDate)
            ->join('emp_company_details', 'leave_requests.employee_id', '=', 'emp_company_details.id')
            ->join('emp_personal_details', 'leave_requests.employee_id', '=', 'emp_personal_details.emp_id')
            ->join('roles', 'emp_company_details.access_role', '=', 'roles.code')
            ->select('roles.title', 'emp_personal_details.*')
            ->get();
    
        return response()->json([
            'employee_details'         => $emp_details,
            'all_leave_packages'       => $all_pkgs,
            'today_on_leave_employees' => $today_on_leave_employees,
            'leave_requests'           => $leave_requests,
        ]);
    }

    public function employee_pkg(Request $request, $id)
    {
        
        $emp_details = EmpCompanyDetails::where('id', $id)->first();
    
        $leave_packages    = LeavePackage::whereRaw("FIND_IN_SET(?, employee_roles)", [$emp_details->accessRole->id])->get();
        
   
        if (!$leave_packages) {
            return response()->json(['error' => 'Leave package not found'], 400);
        }
    

        return response()->json([
            'all_packages' => $leave_packages
        ], 200);
    }
    public function leave_count(Request $request, $id)
    {
        $leavePackage = LeavePackage::find($id);
    
        if (!$leavePackage) {
            return response()->json(['error' => 'Leave package not found'], 404);
        }
    
        $remaining_leave = $leavePackage->leave_number;
        $leave_title = $leavePackage->title;
        return response()->json([
            'remaining_leave' => $remaining_leave,
            'title_leave' => $leave_title
        ], 200);
    }

    public function leave_pkg(Request $request)
    {
        $leave = LeavePackage::all();
    
        if (!$leave) {
            return response()->json(['error' => 'Leave packages not found'], 404);
        }
    
        return response()->json([
            'message' => 'All Leave Packages',
            'data' => $leave
        ], 200);
    }


    
    public function leave_portal(Request $request)
    {
        $current_date = date('Y-m-d');
        // $all_emp = EmpCompanyDetails::where('status',1)->where('compeleted',1)->where('del',0)->count();
        $employee_ids = RosterAssign::join('emp_company_details','roster_assigns.assign_to','emp_company_details.id')
                    ->where('emp_company_details.del',0)->where('emp_company_details.status',1)
                    ->where('roster_assigns.schedule_date', $current_date)
                    ->groupBy('roster_assigns.assign_to')
                    ->pluck('roster_assigns.assign_to');
        $all_emp = count($employee_ids);
        $present_employee_ids = EmployeeAttendance::where('date',$current_date)->wherein('employee_id',$employee_ids)
                                    ->groupBy('employee_id')->pluck('employee_id')->toArray();
        $present_employees = count($present_employee_ids);

        $leave_employee_ids = LeaveRequest::wherein('employee_id',$employee_ids)->where('from',"=",$current_date)
                                    ->where('status',1)->pluck('employee_id')->toArray();

        $leave_employees = count($leave_employee_ids);
       
        return view('LeaveRequest.leaveportal', compact('all_emp','leave_employees','present_employees'));
    }

    public function upcoming_leave(Request $request)
    {
        $today = Carbon::today();

        $query = LeaveRequest::query();
        $query->where('status','0');
        $query->where('from', '>=', $today);
        if($request->filled('filter'))
        {
          $query =  $this->filter(json_decode($request->filter, true),$query);
        }
        $query->with('LeavePackage')->with('EmpPersonalDetails');
        $count = $query->count();
        $start = $request->input('from', 0);
        $query->offset($start)->limit(10);
        $query->orderBy('id', 'desc');
        $query_result = $query->get();
        $i = 0;

        foreach ($query_result as $query_resul) {
            

            $from = Carbon::parse($query_resul->from);
            $to = Carbon::parse($query_resul->to);
           
            $days = $to->diffInDays($from) + 1;
        
           
            $query_result[$i]->total_day = $days;
            $i++;
        }
        $data = [
            'all_leaves' => $query_result,
        ];
        
        $array_filter =  json_decode($request->filter, true);
       

        return response()->json([
            'message' => 'Get Leave Package List Successfully',
            'data' => $data,
            'count' => $count
        ],200);
    }

    
    public function approved_leave(Request $request)
    {
        $query = LeaveRequest::query();
        $query->where('status','1');
        if($request->filled('filter'))
        {
          $query =  $this->filter(json_decode($request->filter, true),$query);
        }
        $query->with('LeavePackage')->with('EmpPersonalDetails');

        $count = $query->count();
        $start = $request->input('from', 0);
        $query->offset($start)->limit(10);
        $query->orderBy('id', 'desc');
        $query_result = $query->get();
        $i = 0;

        foreach ($query_result as $query_resul) {
            

            $from = Carbon::parse($query_resul->from);
            $to = Carbon::parse($query_resul->to);
           
            $days = $to->diffInDays($from) + 1;
        
           
            $query_result[$i]->total_day = $days;
            $i++;
        }
        $data = [
            'approve_leaves' => $query_result,
        ];
        
        $array_filter =  json_decode($request->filter, true);
       

        return response()->json([
            'message' => 'Get Leave Package List Successfully',
            'data' => $data,
            'count' => $count
        ],200);
    }
    public function rejected_leave(Request $request)
    {
        $query = LeaveRequest::query();
        $query->where('status','2');
        if($request->filled('filter'))
        {
          $query =  $this->filter(json_decode($request->filter, true),$query);
        }
        $query->with('LeavePackage')->with('EmpPersonalDetails');

        $count = $query->count();
        $start = $request->input('from', 0);
        $query->offset($start)->limit(10);
        $query->orderBy('id', 'desc');
        $query_result = $query->get();

        $i = 0;

        foreach ($query_result as $query_resul) {
            

            $from = Carbon::parse($query_resul->from);
            $to = Carbon::parse($query_resul->to);
           
            $days = $to->diffInDays($from) + 1;
        
           
            $query_result[$i]->total_day = $days;
            $i++;
        }
        

        $data = [
            'rejected_leaves' => $query_result
        ];
       
        $array_filter =  json_decode($request->filter, true);
       

        return response()->json([
            'message' => 'Get Leave Package List Successfully',
            'data' => $data,
            'count' => $count
        ],200);
    }
   
    public function exipired_leave(Request $request)
    {
        $currentDate = Carbon::now();

        $query = LeaveRequest::query();
        $query->where('status','3');
        if($request->filled('filter'))
        {
          $query =  $this->filter(json_decode($request->filter, true),$query);
        }
        $query->with('LeavePackage')->with('EmpPersonalDetails');

        $count = $query->count();
        $start = $request->input('from', 0);
        $query->offset($start)->limit(10);
        $query->orderBy('id', 'desc');
        $query_result = $query->get();
        $i = 0;

        foreach ($query_result as $query_resul) {
            

            $from = Carbon::parse($query_resul->from);
            $to = Carbon::parse($query_resul->to);
           
            $days = $to->diffInDays($from) + 1;
        
           
            $query_result[$i]->total_day = $days;
            $i++;
        }
        $data = [
            'expired_leaves' => $query_result,
        ];
        
        $array_filter =  json_decode($request->filter, true);
       

        return response()->json([
            'message' => 'Get Leave Package List Successfully',
            'data' => $data,
            'count' => $count
        ],200);
    }


    public function update_expired_status()
    {
        $expiredLeaves = LeaveRequest::where('to', '<', Carbon::now())
            ->where('status', 0)
            ->where('status', '!=', 3) 
            ->get();

       foreach ($expiredLeaves as $leave) { $leave->update(['status' => 3]); }
       return 0;
    }
    public function approveMultiple(Request $request)
    {
        $validated = $request->validate([
            'ids'    => 'required|array',
            'ids.*'  => 'integer|exists:leave_requests,id',
            'status' => 'required|in:1,2', // 1=Approved, 2=Rejected
        ]);

        try {
            $updatedCount = LeaveRequest::whereIn('id', $validated['ids'])
                ->update(['status' => $validated['status']]);

            $statusText = $validated['status'] == 1 ? 'approved' : 'rejected';

            return response()->json([
                'success' => true,
                'message' => $updatedCount . ' leave request(s) ' . $statusText . ' successfully.',
                'updated_count' => $updatedCount
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating leave requests: ' . $e->getMessage()
            ], 500);
        }
    }

    public function customApprove(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'paid_days' => 'required|integer|min:0',
            'reason' => 'nullable|string',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'message' => $validator->errors()->first(),
            ], 422);
        }

        $leaveRequest = LeaveRequest::findOrFail($id);
        $from = Carbon::parse($leaveRequest->from);
        $to = Carbon::parse($leaveRequest->to);
        $totalDays = $from->diffInDays($to) + 1;

        $paidDaysRequested = (int) $request->input('paid_days');
        if ($paidDaysRequested > $totalDays) {
            return response()->json([
                'message' => 'Paid days cannot exceed total leave days ('.$totalDays.').',
            ], 422);
        }

        // Build array of leave dates and pick earliest N as paid dates
        $period = new \DatePeriod($from, new \DateInterval('P1D'), $to->copy()->addDay());
        $allDates = [];
        foreach ($period as $date) {
            $allDates[] = $date->format('Y-m-d');
        }
        $paidDates = array_slice($allDates, 0, $paidDaysRequested);

        $leaveRequest->status = 1; // approved
        $leaveRequest->approved_by = Auth::user()->id ?? null;
        $leaveRequest->paid_days = $paidDaysRequested;
        $leaveRequest->paid_dates = json_encode($paidDates);
        $leaveRequest->custom_approval_reason = $request->input('reason');
        $leaveRequest->save();

        // Optional: send same email as in updateStatus when approved
        $leave_pkg = LeavePackage::where('id', $leaveRequest->leave_package_id)->first();
        $emp_mail = EmpCompanyDetails::where('id', $leaveRequest->employee_id)->first();
        $emp_detail = EmpPersonalDetails::where('emp_id', $leaveRequest->employee_id)->first();

        $data = [
            'status' => $leaveRequest->status,
            'name' => ($emp_detail->first_name ?? '') . ' ' . ($emp_detail->middle_name ?? '') . ' ' . ($emp_detail->last_name ?? ''),
            'subject' => ($leave_pkg->title ?? 'Leave').' | '.env('APP_NAME'),
        ];

        if ($emp_mail && $emp_mail->employee_email) {
            $this->sendEmail(
                $emp_mail->employee_email,
                ($leave_pkg->title ?? 'Leave').' | '.env('APP_NAME'),
                $data,
                'Emails.leave-email'
            );
        }

        return response()->json(['message' => 'Leave custom-approved successfully.']);
    }
}
