<?php

namespace App\Http\Controllers;

use Carbon\Carbon;
use App\Models\Sites;
use App\Models\Overtime;
use Illuminate\Http\Request;
use App\Models\EmpCompanyDetails;

class OvertimeController extends Controller
{
    public function index()
    {
        $overtimes = Overtime::all();  // Get all overtime entries
        return view('overtimes.index', compact('overtimes'));  // Return view with data
    }

    // Show the form for creating a new overtime entry
    public function create()
    {
       
        $get_employes =  EmpCompanyDetails::with('EmpPersonalDetails')->where('del',0)->where('compeleted', 1)->orderBy('id', 'DESC')->get();
        $sites = Sites::where('del',0)->where('active',1)->get();
        return view('overtimes.create',compact('get_employes','sites'));
    }

    // Store a newly created overtime entry in storage
    public function store(Request $request)
    {
        $validatedData = $request->validate([
            'employee_id'   => 'required|integer',
            'site_id'       => 'required|integer',
            'check_in'      => 'required|date',
            'check_out'     => 'required|date|after:check_in',
            'date'          => 'required|date',
            'status'        => 'nullable|string',
            'description'   => 'required|string',
        ]);
    
        // Check if an overtime record already exists for this employee with the same check in and check out times.
        $existingOvertime = Overtime::where('employee_id', $validatedData['employee_id'])
        ->where(function ($query) use ($validatedData) {
            $query->whereBetween('check_in', [$validatedData['check_in'], $validatedData['check_out']])
                  ->orWhereBetween('check_out', [$validatedData['check_in'], $validatedData['check_out']])
                  ->orWhere(function($query) use ($validatedData) {
                      $query->where('check_in', '<', $validatedData['check_in'])
                            ->where('check_out', '>', $validatedData['check_out']);
                  });
        })
        ->first();
    
    
        if ($existingOvertime) {
            return response()->json([
                'status'  => 'error',
                'message' => 'An overtime record with the same check-in and check-out times already exists for this employee!'
            ], 400);
        }
    
        $checkIn = Carbon::parse($validatedData['check_in']);
        $checkOut = Carbon::parse($validatedData['check_out']);
        $minutes = $checkIn->diffInMinutes($checkOut);
    
        $validatedData['working_hours'] = $minutes;
    
        Overtime::create($validatedData);
    
        return response()->json([
            'status'  => 'success',
            'message' => 'Overtime Created Successfully!'
        ]);
    }
    


    public function getOvertimes(Request $request)
    {
        // Start building the query on the Overtime model
        $query = Overtime::query();
    
        // If a filter is provided, apply it using a custom filter method
        if ($request->filled('filter')) {
            $query = $this->filter(json_decode($request->filter, true), $query);
        }
    
        // Eager load relationships similar to your attendance query.
        // Adjust relationship names as defined in your Overtime model.
        $query->with(['emppersonaldetails', 'empcompanydetails', 'empTeamsMembers', 'sites']);
    
        // Get the total count for pagination purposes
        $count = $query->count();
    
        // Get the starting offset (defaults to 0)
        $start = $request->input('from', 0);
    
        // Apply offset, limit, and order
        $query->offset($start)->limit(10);
        $query->orderBy('id', 'desc');
    
        // Retrieve the overtime records
        $query_result = $query->get();
    
        $data = [
            'overtimes' => $query_result,
        ];

        $array_filter = json_decode($request->filter, true);
        $filters = [
            'employee_id' => $array_filter['employee_id'] ?? '',
            'site_id'     => $array_filter['site_id'] ?? '',
            'status'      => $array_filter['status'] ?? '',
        ];
    
        // Return the JSON response
        return response()->json([
            'message' => 'Get Overtime List Successfully',
            'data'    => $data,
            'count'   => $count,
            'filters' => $filters,
        ], 200);
    }
    

    public function filter($filters, $query)
    {
        foreach ($filters as $filterName => $filterValue) {
            if ($filterValue != null || $filterValue != "") {
                switch ($filterName) {
                    case 'first_name':
                        $filterValue = explode(" ", $filterValue);

                        if (count($filterValue) == 1) {
                            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%')
                                         ->orWhere('middle_name', 'like', '%' . $filterValue[0] . '%')
                                         ->orWhere('last_name', 'like', '%' . $filterValue[0] . '%');
                            });
                        }

                        if (count($filterValue) == 2) {
                            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%')
                                         ->where('middle_name', 'like', '%' . $filterValue[1] . '%')
                                         ->where('last_name', 'like', '%' . $filterValue[1] . '%');
                            });
                        }

                        if (count($filterValue) == 3) {
                            $query->whereHas('empPersonalDetails', function ($subquery) use ($filterValue) {
                                $subquery->where('first_name', 'like', '%' . $filterValue[0] . '%')
                                         ->where('middle_name', 'like', '%' . $filterValue[1] . '%')
                                         ->where('last_name', 'like', '%' . $filterValue[2] . '%');
                            });
                        }
                        break;
                    case 'teams':
                        // Adjust the model namespace as necessary.
                        $emp_teams_members = \App\Models\EmpTeamsMember::where('team_id', $filterValue)
                                                                      ->pluck('emp_id')
                                                                      ->toArray();
                        $query->whereIn('employee_id', $emp_teams_members);
                        break;
                        case 'status':
                            // Just do a direct match (or 'like' if you need partial matches).
                            $query->where('status', $filterValue);
                            break;
                    case 'title':
                        $query->whereHas('sites', function ($subquery) use ($filterValue) {
                            $subquery->where('title', 'like', '%' . $filterValue . '%');
                        });
                        break;
                    case 'check_in':
                        $query->where('check_in', 'like', '%' . $filterValue . '%');
                        break;
                    case 'check_out':
                        $query->where('check_out', 'like', '%' . $filterValue . '%');
                        break;
                    case 'working_hours':
                        $query->where('working_hours', 'like', '%' . $filterValue . '%');
                        break;
                    case 'date':
                        $filterValue = explode("/", $filterValue);
                        $query->whereBetween('date', [
                            date("Y-m-d", strtotime(trim($filterValue[0], " "))),
                            date("Y-m-d", strtotime(trim($filterValue[1], " ")))
                        ]);
                        break;
                    case 'system_checkout':
                        if ($filterValue != 2) {
                            $query->where('checkout_type', $filterValue);
                        }
                        break;
                }
            }
        }

        return $query;
    }

    // Show a specific overtime entry
    public function show($id)
    {
        $overtime = Overtime::findOrFail($id);  // Get a single overtime entry by ID
        return view('overtimes.show', compact('overtime'));  // Return view for show
    }

    // Show the form for editing an overtime entry
    public function edit($id)
    {
        $get_employes =  EmpCompanyDetails::with('EmpPersonalDetails')->where('del',0)->where('compeleted', 1)->orderBy('id', 'DESC')->get();
        $sites = Sites::where('del',0)->where('active',1)->get();
        $overtime = Overtime::findOrFail($id);  // Get the overtime entry by ID
        return view('overtimes.edit', compact('overtime','get_employes','sites'));  // Return edit form
    }

    // Update the specified overtime entry in storage
    public function update(Request $request, $id)
    {
        // Validate the request data. The 'after:check_in' rule ensures that check_out is greater than check_in.
        $validatedData = $request->validate([
            'employee_id'   => 'required|integer',
            'site_id'       => 'required|integer',
            'check_in'      => 'required|date',
            'check_out'     => 'required|date|after:check_in',
            'date'          => 'required|date',
            'status'        => 'nullable|string',
            'description'   => 'required|string',

        ]);
    
        // Check for overlapping overtime entry, excluding the current record.
        $existingOvertime = Overtime::where('employee_id', $validatedData['employee_id'])
            ->where('id', '!=', $id) // Exclude the current overtime record.
            ->where(function ($query) use ($validatedData) {
                $query->whereBetween('check_in', [$validatedData['check_in'], $validatedData['check_out']])
                      ->orWhereBetween('check_out', [$validatedData['check_in'], $validatedData['check_out']])
                      ->orWhere(function($query) use ($validatedData) {
                          $query->where('check_in', '<', $validatedData['check_in'])
                                ->where('check_out', '>', $validatedData['check_out']);
                      });
            })
            ->first();
    
        if ($existingOvertime) {
            return response()->json([
                'success' => false,
                'message' => 'An overlapping overtime entry already exists for this employee!'
            ], 400);
        }
    
        // Calculate working hours (in minutes) from check_in and check_out.
        $checkIn = Carbon::parse($validatedData['check_in']);
        $checkOut = Carbon::parse($validatedData['check_out']);
        $minutes = $checkIn->diffInMinutes($checkOut);
    
        // Add the calculated working minutes to the validated data.
        $validatedData['working_hours'] = $minutes;
    
        // Find the overtime entry by its ID.
        $overtime = Overtime::findOrFail($id);
    
        // Update the overtime entry with the validated and modified data.
        $overtime->update($validatedData);
    
        // Return a JSON response with a success message and the updated overtime data.
        return response()->json([
            'success' => true,
            'message' => 'Overtime updated successfully!',
            'data'    => $overtime
        ]);
    }
    

    //update status
    public function updateStatus(Request $request)
    {
        // Validate the incoming data
        $validated = $request->validate([
            'id'     => 'required|integer|exists:overtimes,id',
            'status' => 'required|in:1,2,3',
        ]);

        // Retrieve and update the overtime record
        $overtime = Overtime::find($request->id);
        $overtime->status = $request->status;
        $overtime->save();

        return response()->json([
            'success' => true,
            'message' => 'Overtime status updated successfully.',
        ]);
    }


    // Remove the specified overtime entry from storage
    public function destroy($id)
    {
        $overtime = Overtime::findOrFail($id);
        $overtime->delete();  // Delete the overtime entry

        return response()->json([
            'message' => 'Overtime deleted successfully!'
        ]);
    }

    public function applyOvertime(Request $request)
    {
        // Validate incoming request data
        $validatedData = $request->validate([
            'check_in'    => 'required|date',
            'check_out'   => 'required|date|after:check_in',
            'date'        => 'required|date',
            'site_id'     => 'required|integer|exists:sites,id',
            'description' => 'required|string',
        ]);

        $validatedData['employee_id'] = auth()->user()->id;

        $existingOvertime = Overtime::where('employee_id', $validatedData['employee_id'])
            ->where(function ($query) use ($validatedData) {
                $query->whereBetween('check_in',  [$validatedData['check_in'], $validatedData['check_out']])
                    ->orWhereBetween('check_out', [$validatedData['check_in'], $validatedData['check_out']])
                    ->orWhere(function ($query) use ($validatedData) {
                        $query->where('check_in',  '<', $validatedData['check_in'])
                                ->where('check_out', '>', $validatedData['check_out']);
                    });
            })
            ->first();

        if ($existingOvertime) {
            return response()->json([
                'status'  => 'error',
                'message' => 'An overtime record with the same or overlapping check-in and check-out times already exists!',
            ], 400);
        }

        $checkIn  = Carbon::parse($validatedData['check_in']);
        $checkOut = Carbon::parse($validatedData['check_out']);
        $totalMinutes = $checkIn->diffInMinutes($checkOut);

        $validatedData['working_hours'] = $totalMinutes;
        $validatedData['status']        = 1; 

   
        Overtime::create($validatedData);

        return response()->json([
            'status'  => 'success',
            'message' => 'Overtime created successfully!',
        ], 200);
    }

    /**
     * Approve multiple overtime records
     */
    public function approveMultiple(Request $request)
    {
        // Validate the incoming data
        $validated = $request->validate([
            'ids'    => 'required|array',
            'ids.*'  => 'integer|exists:overtimes,id',
            'status' => 'required|in:1,2,3',
        ]);

        try {
            // Update all the overtime records
            $updatedCount = Overtime::whereIn('id', $validated['ids'])
                ->update(['status' => $validated['status']]);

            $statusText = $validated['status'] == 1 ? 'pending' : ($validated['status'] == 2 ? 'approved' : 'rejected');

            return response()->json([
                'success' => true,
                'message' => $updatedCount . ' overtime record(s) ' . $statusText . ' successfully.',
                'updated_count' => $updatedCount
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error updating overtime records: ' . $e->getMessage()
            ], 500);
        }
    }


}
